﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using MyStoreIntegration.MyStore;
using System.IO;

namespace MyStoreIntegration.Integration
{
    class InventoryDataRetriever
    {
        //Retrieving the list of stock items
        public static void ExportStockItems(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Retrieving the list of stock items...");

            //Specify the parameters of stock items to be returned
            StockItem stockItemsToBeFound = new StockItem
            {
                //Specify return behavior
                ReturnBehavior = ReturnBehavior.OnlySpecified,

                //Filter the items by the last modified date and status
                LastModified = new DateTimeSearch
                {
                    Value = DateTime.Now.AddMonths(-1),
                    Condition = DateTimeCondition.IsGreaterThan
                },
                ItemStatus = new StringSearch { Value = "Active" },

                //Specify other fields to be returned
                InventoryID = new StringReturn(),
                Description = new StringReturn(),
                ItemClass = new StringReturn(),
                BaseUOM = new StringReturn(),
            };

            //Get the list of stock items
            Entity[] stockItems = soapClient.GetList(stockItemsToBeFound);

            //Save results to a CSV file
            using (StreamWriter file = new StreamWriter("StockItems.csv"))
            {
                //Add headers to the file
                file.WriteLine("InventoryID;Description;ItemClass;BaseUOM;LastModified;");

                //Write the values for each item
                foreach (StockItem stockItem in stockItems)
                {
                    file.WriteLine(string.Format("{0};{1};{2};{3};{4};",
                        stockItem.InventoryID.Value,
                        stockItem.Description.Value,
                        stockItem.ItemClass.Value,
                        stockItem.BaseUOM.Value,
                        stockItem.LastModified.Value));
                }
            }
        }

        //Retrieving the list of sales categories
        public static void ExportSalesCategories(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Retrieving the list of item sales categories...");

            //Filter the item sales categories, if needed. In this example, no filtering is applied.
            ItemSalesCategory itemCategory = new ItemSalesCategory();

            //Get the list of item sales categories
            Entity[] itemSalesCategories = soapClient.GetList(itemCategory);

            //Save results to a CSV file
            using (StreamWriter file = new StreamWriter("ItemSalesCategories.csv"))
            {
                //Add headers to the file
                file.WriteLine("CategoryID;Description;ParentCategoryID;Path;");

                //Write the values for each item sales category
                foreach (ItemSalesCategory category in itemSalesCategories)
                {
                    file.WriteLine(string.Format("{0};{1};{2};{3};",
                        category.CategoryID.Value,
                        category.Description.Value,
                        category.ParentCategoryID.Value,
                        category.Path.Value));
                }
            }
        }

        //Retrieving the list of stock items by a sales category
        public static void ExportStockItemsByCategory(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Retrieving the list of stock items in a sales category...");

            //Sales category ID
            int categoryID = 12;

            //Specify the needed category
            ItemSalesCategory itemCategoryToBeFound = new ItemSalesCategory
            {
                CategoryID = new IntSearch { Value = categoryID }
            };

            //Get the item sales category
            ItemSalesCategory itemSalesCategry = (ItemSalesCategory)soapClient.Get(itemCategoryToBeFound);

            //Save results to a CSV file
            using (StreamWriter file = new StreamWriter(string.Format(@"StockItems_Category_{0}.csv", itemSalesCategry.Description.Value)))
                if (itemSalesCategry.Members != null)
                {
                    //Add headers to the file
                    file.WriteLine("CategoryID;Description;ParentCategoryID;InventoryID;");

                    //Write the values for each stock item in the category
                    foreach (ItemSalesCategoryMember member in itemSalesCategry.Members)
                    {
                        file.WriteLine(string.Format("{0};{1};{2};{3};",
                            //Category summary
                            itemSalesCategry.CategoryID.Value,
                            itemSalesCategry.Description.Value,
                            itemSalesCategry.ParentCategoryID.Value,
                            //Inventory ID of a category member
                            member.InventoryID.Value));
                    }
                }
        }

        //Retrieving the quantities of a stock item
        public static void ExportItemQty(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Retrieving the quantities of a stock item...");

            //Parameters of filtering
            string inventoryID = "AALEGO500"; 
            string warehouse = "MAIN";

            //Filter warehouse details by warehouse 
            InventorySummaryInquiry stockItemsToBeFound = new InventorySummaryInquiry
            {
                InventoryID = new StringValue { Value = inventoryID },
                WarehouseID = new StringValue { Value = warehouse }
            };

            //Retrieve the list of stock items from the inquiry
            InventorySummaryInquiry stockItems = (InventorySummaryInquiry)soapClient.Put(stockItemsToBeFound);

            //Save results to a CSV file
            using (StreamWriter file = new StreamWriter(string.Format(@"StockItem_Qty_InventoryID_{0}.csv", inventoryID)))
            {
                //Add headers to the file
                file.WriteLine("InventoryID;WarehouseID;OnHand;Available;AvailableforShipment");

                //Write the values for each stock item
                foreach (InventorySummaryRow stockItem in stockItems.Results)
                {
                    //Filter warehouse details by quantity on hand
                    if (stockItem.OnHand.Value > 0)
                        file.WriteLine(string.Format("{0};{1};{2};{3};{4};",
                            stockItems.InventoryID.Value,
                            stockItems.WarehouseID.Value,
                            stockItem.OnHand.Value,
                            stockItem.Available.Value,
                            stockItem.AvailableforShipment.Value));

                }
            }
        }

        //Retrieving the files that are attached to a stock item
        public static void ExportStockItemFiles(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Retrieving the files that are attached to a stock item...");

            //Parameters of filtering
            string inventoryID = "AAMACHINE1";

            //Filter the items by inventory ID
            StockItem stockItemToBeFound = new StockItem
            {
                InventoryID = new StringSearch { Value = inventoryID }
            };

            //Get the stock item record
            StockItem stockItem = (StockItem)soapClient.Get(stockItemToBeFound);

            //Get the files that are attached to the stock item and save them on a local disc
            if (stockItem != null && stockItem.ImageUrl != null)
            {
                //Get the attached files
                MyStore.File[] files = soapClient.GetFiles(stockItem);

                //Save the files on disc
                foreach (MyStore.File file in files)
                {
                    //The file name obtained from Acumatica ERP has the following format:
                    //Stock Items (<Inventory ID>)\<File Name>
                    string fileName = Path.GetFileName(file.Name);
                    System.IO.File.WriteAllBytes(fileName, file.Content);
                }
            }
        }

        //Retrieving the price of a stock item
        public static void GetStockItemPrice(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Retrieving the price of a stock item...");

            //input parameters
            string customer = "C000000003";
            string inventoryID = "AALEGO500";
            //Specify the stock item and customer
            GetSalesPriceInquiry priceToBeGet = new GetSalesPriceInquiry
            {
                Customer = new StringValue { Value = customer },
                InventoryID = new StringValue { Value = inventoryID }
            };

            //Get the price of the item
            GetSalesPriceInquiry stockItemPrice = (GetSalesPriceInquiry)soapClient.Put(priceToBeGet);

            //Display the price of the item
            Console.WriteLine("Stock item price: " + stockItemPrice.Price.Value);
            Console.WriteLine();
            Console.WriteLine("Press any key to continue");
            Console.ReadLine();
        }
    }
}
