﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using MyStoreIntegration.MyStore;
using System.IO;
using MyStoreIntegration.Helpers;

namespace MyStoreIntegration.Integration
{
    class PaymentProcessor
    {
        //Creating a payment 
        public static void CreatePayment(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Creating a payment...");

            //Payment data
            string paymentType = "Payment";
            string customerID = "C000000003";
            string paymentMethod = "VISA"; //A credit card payment method
            //that is configured on the Payment Methods form(CA204000)
            string invoiceNbr = "INV000045"; //Invoice with Open status

            //Specify the parameters of the new payment
            Payment soPaymentToBeCreated = new Payment
            {
                Type = new StringValue { Value = paymentType },
                CustomerID = new StringValue { Value = customerID },
                PaymentMethod = new StringValue { Value = paymentMethod },
                DocumentsToApply = new[]
                        {
                            new PaymentDetail
                            {
                                ReferenceNbr = new StringValue { Value = invoiceNbr }
                            }
                        }

            };

            //Create the payment
            Payment newPayment = (Payment)soapClient.Put(soPaymentToBeCreated);
            newPayment.PaymentAmount = newPayment.AppliedToDocuments;
            newPayment = (Payment)soapClient.Put(newPayment);

            //Display the summary of the created payment
            Console.WriteLine("Payment type: " + newPayment.Type.Value);
            Console.WriteLine("Payment number: " + newPayment.ReferenceNbr.Value);
            Console.WriteLine("Payment status: " + newPayment.Status.Value);
            Console.WriteLine("Payment amount: " + newPayment.PaymentAmount.Value);
            Console.WriteLine();
            Console.WriteLine("Press any key to continue");
            Console.ReadLine();
        }

        //Capturing a payment 
        public static void CapturePayment(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Capturing a credit card payment...");

            //Payment data
            string paymentType = "Payment";
            string paymentNbr = "000001"; //A payment that has On Hold status

            //Find the payment that should be captured
            Payment soPaymentToBeCaptured = new Payment
            {
                Type = new StringSearch { Value = paymentType },
                ReferenceNbr = new StringSearch { Value = paymentNbr },
                Hold = new BooleanValue { Value = false }
            };
            Payment payment = (Payment)soapClient.Put(soPaymentToBeCaptured);

            //Capture payment
            InvokeResult invokeResult = soapClient.Invoke(payment, new CaptureCreditCardPayment());

            //Monitor the status of the process
            ProcessResult processResult = LongRunProcessor.GetProcessResult(soapClient, invokeResult);

            //Get the captured payment
            payment = (Payment)soapClient.Get(new Payment { ID = processResult.EntityId });
            //Display info on the captured payment
            Console.WriteLine("Payment type: " + payment.Type.Value);
            Console.WriteLine("Payment number: " + payment.ReferenceNbr.Value);
            Console.WriteLine("Payment status: " + payment.Status.Value);
            Console.WriteLine("Transaction status: " + payment.CreditCardProcessingInfo[0].TranStatus.Value);
            Console.WriteLine("Transaction amount: " + payment.CreditCardProcessingInfo[0].TranAmount.Value);
            Console.WriteLine();
            Console.WriteLine("Press any key to continue");
            Console.ReadLine();
        }

        //Releasing a payment
        public static void ReleasePayment(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Releasing a payment...");

            //Payment data
            string paymentType = "Payment";
            string paymentNbr = "000001"; //A payment that has Balanced status

            //Find the payment that should be released
            Payment soPaymentToBeReleased = new Payment
            {
                Type = new StringSearch { Value = paymentType },
                ReferenceNbr = new StringSearch { Value = paymentNbr }
            };
            Payment payment = (Payment)soapClient.Get(soPaymentToBeReleased);

            //Release payment
            InvokeResult invokeResult = soapClient.Invoke(payment, new ReleasePayment());

            //Monitor the status of the process
            ProcessResult processResult = LongRunProcessor.GetProcessResult(soapClient, invokeResult);

            //Get the released payment
            payment = (Payment)soapClient.Get(new Payment { ID = processResult.EntityId });
            //Display a summary of the released payment
            Console.WriteLine("Payment type: " + payment.Type.Value);
            Console.WriteLine("Payment number: " + payment.ReferenceNbr.Value);
            Console.WriteLine("Payment status: " + payment.Status.Value);
            Console.WriteLine();
            Console.WriteLine("Press any key to continue");
            Console.ReadLine();
        }

        //Retrieving the list of payments of a customer
        public static void ExportPayments(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Getting the list of payments of a customer...");

            //Input data
            string customerID = "C000000003";
            string docType = "Payment";

            //Select the payments that should be exported
            Payment soPaymentsToBeFound = new Payment
            {
                ReturnBehavior = MyStore.ReturnBehavior.OnlySpecified,

                Type = new StringSearch { Value = docType },
                CustomerID = new StringSearch { Value = customerID },

                ReferenceNbr = new StringReturn(),
                ApplicationDate = new DateTimeReturn(),
                Status = new StringReturn(),
            };
            Entity[] payments = soapClient.GetList(soPaymentsToBeFound);

            //Save results to a CSV file
            using (StreamWriter file = new StreamWriter(string.Format(@"Payments_Customer_{0}.csv", customerID)))
            {
                //Add headers to the file
                file.WriteLine("Type;ReferenceNbr;ApplicationDate;Status;");

                //Write the values for each payment
                foreach (Payment payment in payments)
                {
                    file.WriteLine(string.Format("{0};{1};{2};{3};",
                        //Document summary
                        payment.Type.Value,
                        payment.ReferenceNbr.Value,
                        payment.ApplicationDate.Value,
                        payment.Status.Value));
                }
            }
        }
    }
}
