﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using MyStoreIntegration.MyStore;
using System.IO;
using MyStoreIntegration.Helpers;

namespace MyStoreIntegration.Integration
{
    class SalesOrderProcessor
    {
        //Creating a sales order
        public static void CreateSO(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Creating a sales order...");

            //Sales order data
            string orderType = "SO";
            string customerID = "C000000003";
            string orderDescription = "Test Sales Order";
            string customerOrder = "SO180-009-28"; //Any unique identifier
            string firstItemInventoryID = "AALEGO500";
            decimal firstItemQuantity = 10;
            string firstItemUOM = "PIECE";
            string secondItemInventoryID = "CONGRILL";
            decimal secondItemQuantity = 1;
            string secondItemUOM = "PIECE";

            //Specify the values of a new sales order
            SalesOrder orderToBeCreated = new SalesOrder
            {
                OrderType = new StringValue { Value = orderType },
                CustomerID = new StringValue { Value = customerID },
                Description = new StringValue { Value = orderDescription },
                CustomerOrder = new StringValue { Value = customerOrder },
                Details = new[]
				        {
					        new SalesOrderDetail
					        {
						        InventoryID = new StringValue {Value = firstItemInventoryID},
                                Quantity = new DecimalValue { Value = firstItemQuantity },
                                UOM = new StringValue { Value = firstItemUOM}
					        },
					        new SalesOrderDetail
					        {
						        InventoryID = new StringValue {Value = secondItemInventoryID},
                                Quantity = new DecimalValue { Value = secondItemQuantity },
                                UOM = new StringValue { Value = secondItemUOM}
					        }
				        }
            };

            //Create a sales order with the specified values
            SalesOrder newOrder = (SalesOrder)soapClient.Put(orderToBeCreated);

            //Display the summary of the created sales order
            Console.WriteLine("Order type: " + newOrder.OrderType.Value);
            Console.WriteLine("Order number: " + newOrder.OrderNbr.Value);
            Console.WriteLine("Ordered quantity: " + newOrder.OrderedQty.Value);
            Console.WriteLine("Order total: " + newOrder.OrderTotal.Value);
            Console.WriteLine();
            Console.WriteLine("Press any key to continue");
            Console.ReadLine();
        }

        //Updating the detail lines of a sales order
        public static void UpdateSO(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Updating a sales order...");

            //Sales order data
            string orderType = "SO";
            string customerOrder = "SO248-563-06"; //A unique value that identifies a sales order
            string firstItemInventoryID = "CONTABLE1";
            string firstItemWarehouse = "MAIN";
            string secondItemInventoryID = "AALEGO500";
            string secondItemWarehouse = "MAIN";
            decimal secondItemQuantity = 4;

            //Find the sales order to be updated
            SalesOrder soToBeFound = new SalesOrder
            {
                OrderType = new StringSearch { Value = orderType },
                CustomerOrder = new StringSearch { Value = customerOrder },
            };
            SalesOrder order = (SalesOrder)soapClient.Get(soToBeFound);

            //Find the line to be deleted and mark it for deletion
            //The Single method makes the program find 
            //the only SalesOrderDetail of order.Details 
            //that has the specified InventoryID and WarehouseID
            SalesOrderDetail orderLine = order.Details.Single(
                orderLineToBeDeleted =>
                    orderLineToBeDeleted.InventoryID.Value == firstItemInventoryID &&
                    orderLineToBeDeleted.WarehouseID.Value == firstItemWarehouse);
            orderLine.Delete = true;

            //Find the line to be updated and update the quantity in it
            orderLine = order.Details.Single(
                orderLineToBeUpdated =>
                    orderLineToBeUpdated.InventoryID.Value == secondItemInventoryID &&
                    orderLineToBeUpdated.WarehouseID.Value == secondItemWarehouse);
            orderLine.Quantity = new DecimalValue { Value = secondItemQuantity };

            //Clear the Hold check box
            order.Hold = new BooleanValue { Value = false };

            //Update the sales order
            order = (SalesOrder)soapClient.Put(order);

            //Display the summary of the updated record
            Console.WriteLine("Order type: " + order.OrderType.Value);
            Console.WriteLine("Order number: " + order.OrderNbr.Value);
            Console.WriteLine("Ordered quantity: " + order.OrderedQty.Value);
            Console.WriteLine("Order total: " + order.OrderTotal.Value);
            Console.WriteLine();
            Console.WriteLine("Press any key to continue");
            Console.ReadLine();
        }


        //Retrieving the details of a sales order by the order number
        public static void ExportSODetails(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Getting detail lines of a sales order...");

            //Sales order data
            string orderType = "SO";
            string orderNbr = "000001";

            //Specify the sales order that should be found
            SalesOrder soToBeFound = new SalesOrder
            {
                OrderType = new StringSearch { Value = orderType },
                OrderNbr = new StringSearch { Value = orderNbr },
            };

            //Get the sales order
            SalesOrder orderLines = (SalesOrder)soapClient.Get(soToBeFound);

            if (orderLines == null)
                Console.WriteLine("No sales order has been found.");
            else
            {
                //Save the results to a CSV file
                using (StreamWriter file = new StreamWriter(string.Format(@"SalesOrder_{0}.csv", orderNbr)))
                {
                    //Add headers to the file
                    file.WriteLine("OrderType;OrderNbr;CustomerID;CustomerOrder;Date;OrderedQty;OrderTotal;TaxTotal;InventoryID;UOM;Quantity;UnitPrice");

                    //Write the values for each detail line
                    foreach (SalesOrderDetail orderLine in orderLines.Details)
                    {
                        file.WriteLine(string.Format("{0};{1};{2};{3};{4};{5};{6};{7};{8};{9};{10};{11}",
                            //Document summary
                            orderLines.OrderType.Value,
                            orderLines.OrderNbr.Value,
                            orderLines.CustomerID.Value,
                            orderLines.CustomerOrder.Value,
                            orderLines.Date.Value,
                            orderLines.OrderedQty.Value,
                            orderLines.OrderTotal.Value,
                            //Totals
                            orderLines.TaxTotal.Value,
                            //Document details
                            orderLine.InventoryID.Value,
                            orderLine.UOM.Value,
                            orderLine.Quantity.Value,
                            orderLine.UnitPrice.Value));
                    }
                }
            }
        }

        //Retrieving the list of sales orders of a customer 
        public static void ExportSalesOrders(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Getting the list of sales orders of a customer...");

            //Customer data
            string customerID = "C000000003";

            //Specify the customer ID of a customer whose sales orders should be exported
            SalesOrder ordersToBeFound = new SalesOrder
            {
                ReturnBehavior = MyStore.ReturnBehavior.OnlySpecified,

                CustomerID = new StringSearch { Value = customerID },

                OrderType = new StringReturn(),
                OrderNbr = new StringReturn(),
                CustomerOrder = new StringReturn(),
                Date = new DateTimeReturn(),
                OrderedQty = new DecimalReturn(),
                OrderTotal = new DecimalReturn(),
            };

            //Get the list of sales orders of the customer
            Entity[] soList = soapClient.GetList(ordersToBeFound);

            //Save results to a CSV file
            using (StreamWriter file = new StreamWriter(string.Format(@"SalesOrders_Customer_{0}.csv", customerID)))
            {
                //Add headers to the file
                file.WriteLine("OrderType;OrderNbr;CustomerID;CustomerOrder;Date;OrderedQty;OrderTotal;");

                //Write the values for each sales order
                foreach (SalesOrder salesOrder in soList)
                {
                    file.WriteLine(string.Format("{0};{1};{2};{3};{4};{5};{6}",
                        //Document summary
                        salesOrder.OrderType.Value,
                        salesOrder.OrderNbr.Value,
                        salesOrder.CustomerID.Value,
                        salesOrder.CustomerOrder.Value,
                        salesOrder.Date.Value,
                        salesOrder.OrderedQty.Value,
                        salesOrder.OrderTotal.Value));
                }
            }
        }
    }
}
