﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using MyStoreIntegration.MyStore;
using System.IO;
using MyStoreIntegration.Helpers;

namespace MyStoreIntegration.Integration
{
    class ShipmentProcessor
    {
        //Creating a shipment 
        public static void CreateShipment(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Creating a shipment...");

            //Shipment data
            string shipmentType = "Shipment";
            string customerID = "C000000003";
            string warehouse = "MAIN";
            string firstOrderNbr = "000004"; //Sales order with the Open status for the specified customer
            string firstOrderType = "SO";
            string secondOrderNbr = "000006"; //Sales order with the Open status for the specified customer
            string secondOrderType = "SO";

            //Find the first sales order to be shipped
            SalesOrder orderToBeFound1 = new SalesOrder
            {
                OrderType = new StringSearch { Value = firstOrderType },
                OrderNbr = new StringSearch { Value = firstOrderNbr }
            };
            SalesOrder orderForShipment1 = (SalesOrder)soapClient.Get(orderToBeFound1);

            //Find the second sales order to be shipped
            SalesOrder orderToBeFound2 = new SalesOrder
            {
                OrderType = new StringSearch { Value = secondOrderType },
                OrderNbr = new StringSearch { Value = secondOrderNbr }
            };
            SalesOrder orderForShipment2 = (SalesOrder)soapClient.Get(orderToBeFound2);

            //Select all stock items from the sales orders for shipment
            List<ShipmentDetail> shipmentDetails = new List<ShipmentDetail>();
            foreach (SalesOrderDetail item in orderForShipment1.Details)
            {
                shipmentDetails.Add(new ShipmentDetail
                {
                    OrderType = orderForShipment1.OrderType,
                    OrderNbr = orderForShipment1.OrderNbr,
                    InventoryID = item.InventoryID,
                    WarehouseID = item.WarehouseID
                });
            }
            foreach (SalesOrderDetail item in orderForShipment2.Details)
            {
                shipmentDetails.Add(new ShipmentDetail
                {
                    OrderType = orderForShipment2.OrderType,
                    OrderNbr = orderForShipment2.OrderNbr,
                    InventoryID = item.InventoryID,
                    WarehouseID = item.WarehouseID
                });
            }

            //Specify the values of a new shipment
            Shipment shipment = new Shipment
            {
                Type = new StringValue { Value = shipmentType },
                CustomerID = new StringValue { Value = customerID },
                WarehouseID = new StringValue { Value = warehouse },
                Details = shipmentDetails.ToArray()
            };

            //Create a shipment with the specified values
            Shipment newShipment = (Shipment)soapClient.Put(shipment);

            //Display the summary of the created record
            Console.WriteLine("Shipment number: " + newShipment.ShipmentNbr.Value);
            Console.WriteLine("Shipment type: " + newShipment.Type.Value);
            Console.WriteLine("Shipment status: " + newShipment.Status.Value);
            Console.WriteLine();
            Console.WriteLine("Press any key to continue");
            Console.ReadLine();
        }

        //Confirming a shipment
        public static void ConfirmShipment(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Confirming a shipment...");

            //Shipment data
            string shipmentNbr = "000001"; //A shipment that has On Hold status
            string shipmentType = "Shipment";

            //Get the shipment to be confirmed 
            Shipment shipmentToBeFound = new Shipment
            {
                Type = new StringSearch { Value = shipmentType },
                ShipmentNbr = new StringSearch { Value = shipmentNbr },
            };
            Shipment shipment = (Shipment)soapClient.Get(shipmentToBeFound);

            //Clear the Hold check box for the shipment
            shipment.Hold = new BooleanValue { Value = false };
            Shipment openShipment = (Shipment)soapClient.Put(shipment);

            //Invoke confirmation
            InvokeResult invokeResult = soapClient.Invoke(openShipment, new ConfirmShipment());

            //Monitor the status of the process
            ProcessResult processResult = LongRunProcessor.GetProcessResult(soapClient, invokeResult);

            //Get the confirmed shipment
            Shipment confirmedShipment = (Shipment)soapClient.Get(new Shipment { ID = processResult.EntityId });
            //Display the summary of the shipment
            Console.WriteLine("Shipment number: " + confirmedShipment.ShipmentNbr.Value);
            Console.WriteLine("Shipment type: " + confirmedShipment.Type.Value);
            Console.WriteLine("Shipment status: " + confirmedShipment.Status.Value);
            Console.WriteLine();
            Console.WriteLine("Press any key to continue");
            Console.ReadLine();
        }

        //Exporting the list of shipments of a customer 
        public static void ExportShipments(DefaultSoapClient soapClient)
        {
            Console.WriteLine("Getting the list of shipments of a customer...");

            //Shipment data
            string customerID = "C000000003";
            string shipmentType = "Shipment";

            //Filter the shipments by type and customer ID
            Shipment shipmentsToBeFound = new Shipment
            {
                ReturnBehavior = MyStore.ReturnBehavior.OnlySpecified,

                Type = new StringSearch { Value = shipmentType },
                CustomerID = new StringSearch { Value = customerID },

                ShipmentNbr = new StringReturn(),
                Status = new StringReturn(),
            };

            //Get the list of shipments
            Entity[] shipments = soapClient.GetList(shipmentsToBeFound);

            //Save the results to a CSV file
            using (StreamWriter file = new StreamWriter(string.Format(@"Shipments_Customer_{0}.csv", customerID)))
            {
                //Add headers to the file
                file.WriteLine("Type;ShipmentNbr;CustomerID;Status;");

                //Write the values of each shipment
                foreach (Shipment shipm in shipments)
                {
                    file.WriteLine(string.Format("{0};{1};{2};{3};",
                        //Document summary
                        shipm.Type.Value,
                        shipm.ShipmentNbr.Value,
                        shipm.CustomerID.Value,
                        shipm.Status.Value));
                }
            }
        }
    }
}
