﻿using System;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Text;
using System.Threading;
using System.IO;
using Newtonsoft.Json;
using MyStoreRestIntegration.Integration;

namespace MyStoreRestIntegration.Helpers
{
    public class RestService : IDisposable
    {
        private readonly HttpClient _httpClient;

        public RestService()
        {
            _httpClient = new HttpClient(
                new HttpClientHandler
                {
                    UseCookies = true,
                    CookieContainer = new CookieContainer()
                })
            {
                BaseAddress = new Uri(Properties.Settings.Default.AcumaticaBaseURL + Properties.Settings.Default.Endpoint),
                DefaultRequestHeaders =
                {
                    Accept = { MediaTypeWithQualityHeaderValue.Parse("text/json") }
                }
            };
        }

        void IDisposable.Dispose()
        {
            _httpClient.Dispose();
        }

        public void Login()
        {
            string credentialsAsString = JsonConvert.SerializeObject(new
            {
                name = Properties.Settings.Default.UserName,
                password = Properties.Settings.Default.Password,
                company = Properties.Settings.Default.CompanyName,
                branch = Properties.Settings.Default.Branch
            });

            var response = _httpClient.PostAsync(Properties.Settings.Default.AcumaticaBaseURL + "/entity/auth/login", new StringContent(credentialsAsString, Encoding.UTF8, "application/json")).Result;
            if (!response.IsSuccessStatusCode)
            {
                Console.Error.WriteLine("HTTP request failed.");
                Console.WriteLine(response);
                Console.WriteLine();
                Console.WriteLine("Press any key to continue");
                Console.ReadLine();
            }

        }

        public void Logout()
        {
            _httpClient.PostAsync(Properties.Settings.Default.AcumaticaBaseURL + "/entity/auth/logout", new ByteArrayContent(new byte[0])).Wait();
        }

        public string Put(string entityName, string entity, string parameters)
        {

            var response = _httpClient
                .PutAsync(_httpClient.BaseAddress + entityName + '?' + parameters, new StringContent(entity, Encoding.UTF8, "application/json")).Result;

            if (!response.IsSuccessStatusCode)
            {
                Console.Error.WriteLine("HTTP request failed:" + response.ReasonPhrase);
                Console.WriteLine(response.Content.ReadAsStringAsync().Result);
                Console.WriteLine();
                Console.WriteLine("Press any key to continue");
                Console.ReadLine();
                return null;
            }
            else return response.Content.ReadAsStringAsync().Result;

        }

        public string Get(string entityName, string parameters)
        {

            var response = _httpClient.GetAsync(_httpClient.BaseAddress + entityName + "?" + parameters).Result;

            if (!response.IsSuccessStatusCode)
            {
                Console.Error.WriteLine("HTTP request failed:" + response.ReasonPhrase);
                Console.WriteLine(response.Content.ReadAsStringAsync().Result);
                Console.WriteLine();
                Console.WriteLine("Press any key to continue");
                Console.ReadLine();
                return null;
            }
            else return response.Content.ReadAsStringAsync().Result;
        }

        public string GetByKeys(string entityName, string keys, string parameters)
        {
            var response = _httpClient.GetAsync(_httpClient.BaseAddress + entityName + "/" + keys + "?" + parameters).Result;

            if (!response.IsSuccessStatusCode)
            {
                Console.Error.WriteLine("HTTP request failed:" + response.ReasonPhrase);
                Console.WriteLine(response.Content.ReadAsStringAsync().Result);
                Console.WriteLine();
                Console.WriteLine("Press any key to continue");
                Console.ReadLine();
                return null;
            }
            else return response.Content.ReadAsStringAsync().Result;
        }

        public void Delete(string entityName, string keys)
        {
            var response = _httpClient.DeleteAsync(_httpClient.BaseAddress + entityName + "/" + keys).Result;

            if (!response.IsSuccessStatusCode)
            {
                Console.Error.WriteLine("HTTP request failed:" + response.ReasonPhrase);
                Console.WriteLine(response.Content.ReadAsStringAsync().Result);
                Console.WriteLine();
                Console.WriteLine("Press any key to continue");
                Console.ReadLine();
            }
        }

        public Stream GetFile(string href)
        {
            var response = _httpClient.GetAsync(href).Result;

            if (!response.IsSuccessStatusCode)
            {
                Console.Error.WriteLine("HTTP request failed:" + response.ReasonPhrase);
                Console.WriteLine(response.Content.ReadAsStringAsync().Result);
                Console.WriteLine();
                Console.WriteLine("Press any key to continue");
                Console.ReadLine();
                return null;
            }
            else return response.Content.ReadAsStreamAsync().Result;
        }

        public void PutFile(string entityName, string keys, string fileName, Stream file)
        {
            var response = _httpClient.PutAsync(_httpClient.BaseAddress + entityName + "/" + keys + "/files/" + fileName, new StreamContent(file)).Result;

            if (!response.IsSuccessStatusCode)
            {
                Console.Error.WriteLine("HTTP request failed:" + response.ReasonPhrase);
                Console.WriteLine(response.Content.ReadAsStringAsync().Result);
                Console.WriteLine();
                Console.WriteLine("Press any key to continue");
                Console.ReadLine();
            }
        }

        public string Post(string entityName, string actionName, string entityAndParameters)
        {
            var response = _httpClient
                .PostAsync(_httpClient.BaseAddress + entityName + "/" + actionName, new StringContent(entityAndParameters, Encoding.UTF8, "application/json")).Result;
            
            //If an error occurs in Acumatica ERP, contains the error message.
            var content = response.Content.ReadAsStringAsync().Result;
            response.EnsureSuccessStatusCode();

            var dt = DateTime.Now;
            while (true)
            {
                switch (response.StatusCode)
                {
                    case HttpStatusCode.NoContent:
                        return "No content";
                    case HttpStatusCode.Accepted:
                        if ((DateTime.Now - dt).Seconds > 30)
                            throw new TimeoutException();
                        Thread.Sleep(500);
                        response = _httpClient.GetAsync(response.Headers.Location).Result.EnsureSuccessStatusCode();
                        continue;
                    default:
                        throw new InvalidOperationException("Invalid process result: " + response.StatusCode);
                }
            }                
        }
    }
}
