﻿using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.IO;
using System.Net;
using System.Collections.Generic;
using MyStoreRestIntegration.Helpers;

namespace MyStoreRestIntegration.Integration
{
    public class InventoryDataRetriever
    {
        //Retrieving the list of stock items
        public static void ExportStockItems(RestService rs)
        {
            //Filtering parameters
            string parameters = "$filter=ItemStatus eq 'Active' and LastModified gt datetimeoffset'" +
               WebUtility.UrlEncode(new DateTimeOffset(new DateTime(2016, 2, 1).AddMonths(-1)).ToString("yyyy-MM-ddTHH:mm:ss.fffK")) + "'";

            //Get the list of stock items
            string stockItems = rs.Get("StockItem", parameters);

            if (stockItems != null)
            {
                List<JObject> stockItemList = JsonConvert.DeserializeObject<List<JObject>>(stockItems);

                //Save results to a CSV file
                using (StreamWriter file = new StreamWriter("REST_StockItems.csv"))
                {
                    //Add headers to the file
                    file.WriteLine(
                        "InventoryID;Description;ItemClass;BaseUOM;LastModified;");

                    //Write the values for each item     
                    foreach (JObject stockItem in stockItemList)
                    {
                        file.WriteLine(string.Format("{0};{1};{2};{3};{4};",
                             stockItem.GetValue("InventoryID").Value<string>("value"),
                             stockItem.GetValue("Description").Value<string>("value"),
                             stockItem.GetValue("ItemClass").Value<string>("value"),
                             stockItem.GetValue("BaseUOM").Value<string>("value"),
                             stockItem.GetValue("LastModified").Value<string>("value")));
                    }
                    file.Flush();
                }
            }        
        }

        //Retrieving the list of sales categories
        public static void ExportSalesCategories(RestService rs)
        {
            //Filter the item sales categories, if needed.
            //In this example, no filtering is applied. 

            //Get the list of item sales categories
            string salesCategories = rs.Get("ItemSalesCategory", null);

            if (salesCategories != null)
            {
                List<JObject> ItemSalesCategoryList = JsonConvert.DeserializeObject<List<JObject>>(salesCategories);

                //Save results to a CSV file
                using (StreamWriter file = new StreamWriter("REST_ItemSalesCategories.csv"))
                {
                    //Add headers to the file
                    file.WriteLine("CategoryID;Description;ParentCategoryID;Path;");

                    //Write the values for each item     
                    foreach (JObject salesCategory in ItemSalesCategoryList)
                    {
                        file.WriteLine(string.Format("{0};{1};{2};{3};",
                            salesCategory.GetValue("CategoryID").Value<string>("value"),
                            salesCategory.GetValue("Description").Value<string>("value"),
                            salesCategory.GetValue("ParentCategoryID").Value<string>("value"),
                            salesCategory.GetValue("Path").Value<string>("value")));
                    }
                    file.Flush();
                }
            }                         
        }

        //Retrieving the list of stock items by a sales category
        public static void ExportStockItemsByCategory(RestService rs)
        {
            //Filter the item sales categories and specify the entities to be expanded in response
            string filter = "$filter=CategoryID eq 12";
            string expand = "$expand=Members";

            //Get the item sales category
            string salesCategoryData = rs.Get("ItemSalesCategory", filter + '&' + expand);

            if (salesCategoryData != null)
            {
                //Get the list of istock items for the sales category            
                JObject salesCategory = JsonConvert.DeserializeObject<dynamic>(salesCategoryData)[0];

                string categoryDescription = salesCategory.GetValue("Description").Value<string>("value");

                //Save results to a CSV file
                using (StreamWriter file = new StreamWriter(string.Format(
                    @"REST_StockItems_Category_{0}.csv", categoryDescription)))
                {
                    //Add headers to the file
                    file.WriteLine(
                        "CategoryID;Description;ParentCategoryID;InventoryID;");

                    //Write the values for each stock item in the category
                    JArray Members = salesCategory.Value<JArray>("Members");
                    foreach (JObject Member in Members)
                    {
                        file.WriteLine(string.Format("{0};{1};{2};{3};",
                            salesCategory.GetValue("CategoryID").Value<string>("value"),
                            categoryDescription,
                            salesCategory.GetValue("ParentCategoryID").Value<string>("value"),
                            Member.GetValue("InventoryID").Value<string>("value")));
                    }
                    file.Flush();
                }
            }        
        }

        //Retrieving the quantities of a stock item
        public static void ExportItemQty(RestService rs)
        {
            //Parameters of filtering
            string inventoryId = "AALEGO500";
            string warehouseId = "MAIN";

            //The entity to be expanded in response
            string parameters = "$expand=Results";

            string entityAsString = JsonConvert.SerializeObject(new
            {
                InventoryID = new { value = inventoryId },
                WarehouseID = new { value = warehouseId }
            });

            //Retrieve the list of stock items from the inquiry
            string inventorySummary = rs.Put("InventorySummaryInquiry", entityAsString, parameters);

            if (inventorySummary != null)
            {

                JObject summary = JObject.Parse(inventorySummary);

                //Save results to a CSV file
                using (StreamWriter file = new StreamWriter(string.Format(
                 @"REST_StockItem_Qty_InventoryID_{0}.csv", inventoryId)))
                {
                    //Add headers to the file
                    file.WriteLine(
                        "InventoryID;WarehouseID;OnHand;Available;AvailableforShipment");

                    //Write the values for each iresult
                    JArray Results = summary.Value<JArray>("Results");

                    //Write the values for each item     
                    foreach (JObject result in Results)
                    {
                        string warehouseID = JsonConvert.DeserializeObject<dynamic>(result.ToString())["WarehouseID"].value;
                        string onHand = JsonConvert.DeserializeObject<dynamic>(result.ToString())["OnHand"].value;
                        string available = JsonConvert.DeserializeObject<dynamic>(result.ToString())["Available"].value;
                        string availableforShipment = JsonConvert.DeserializeObject<dynamic>(result.ToString())["AvailableforShipment"].value;

                        file.WriteLine(string.Format("{0};{1};{2};{3};{4};",
                            summary.GetValue("InventoryID").Value<string>("value"),
                            result.GetValue("WarehouseID").Value<string>("value"),
                            result.GetValue("OnHand").Value<string>("value"),
                            result.GetValue("Available").Value<string>("value"),
                            result.GetValue("AvailableforShipment").Value<string>("value")));
                    }
                    file.Flush();
                }
            }        
        }

        //Retrieving the files that are attached to a stock item
        public static void ExportStockItemFiles(RestService rs)
        {
            //Filter the items by inventory ID                   
            string inventoryID = "AAMACHINE1";

            //Retrieve the stock item
            string stockItem = rs.GetByKeys("StockItem", inventoryID, null);

            if (stockItem != null)
            {
                //Find href and the file name of the needed file
                //(using Newtonsoft.Json.Linq and System.IO)
                JObject jItem = JObject.Parse(stockItem);
                JArray jFiles = jItem.Value<JArray>("files");
                foreach (JToken jFile in jFiles)
                {
                    string fileRef = jFile.Value<string>("href");
                    string fullFileName = jFile.Value<string>("filename");
                    string fileName = Path.GetFileName(fullFileName);

                    //Obtain the file
                    Stream file = rs.GetFile(fileRef);
                    if (file != null)
                    {
                        using (var outputFile = File.Create(fileName))
                        {
                            file.Seek(0, SeekOrigin.Begin);
                            file.CopyTo(outputFile);
                        }
                    }
                }
            }        
        }

        //Retrieving the price of a stock item
        public static void GetStockItemPrice(RestService rs)
        {
            //input parameters
            string customer = "C000000003";
            string inventoryID = "AALEGO500";

            //Convert the data to JSON format
            string salesPriceAsString = JsonConvert.SerializeObject(new
            {
                Customer = new { value = customer },
                InventoryID = new { value = inventoryID }
            });

            //Get the price of the item
            string salesPrice = rs.Put("GetSalesPriceInquiry", salesPriceAsString, null);


            //Display the price of the item
            decimal? price = JsonConvert.DeserializeObject<dynamic>(salesPrice)["Price"].value;
            Console.WriteLine("Stock item price: " + price);
            Console.WriteLine();
            Console.WriteLine("Press any key to continue");
            Console.ReadLine();
        }
    }
}
