﻿using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using MyStoreRestIntegration.Helpers;

namespace MyStoreRestIntegration.Integration
{
    public class PaymentProcessor
    {
        //Creating a payment
        public static void CreatePayment(RestService rs)
        {
            //Payment data
            string paymentType = "Payment";
            string customerID = "C000000003";
            string paymentMethod = "VISA"; //A credit card payment method,
            //which is configured on the Payment Methods form(CA204000)
            //int cardAccountNo = 0027; //One of the available
            //cards of the customer
            string invoiceNbr = "INV000045"; //Invoice with Open status

            string paymentAsString = JsonConvert.SerializeObject(new
            {
                Type = new { value = paymentType },
                CustomerID = new { value = customerID },
                PaymentMethod = new { value = paymentMethod },
                DocumentsToApply = new[]
                        {
                            new
                            {
                                ReferenceNbr = new  {value = invoiceNbr}
                            }
                        }
            });

            //Create the payment
            string paymentCreated = rs.Put("Payment", paymentAsString, null);
            if (paymentCreated != null)
            {

                paymentAsString = JsonConvert.SerializeObject(new
                {
                    Type = new { value = paymentType },
                    ReferenceNbr = new { value = JsonConvert.DeserializeObject<dynamic>(paymentCreated)["ReferenceNbr"].value },
                    PaymentAmount = new { value = JsonConvert.DeserializeObject<dynamic>(paymentCreated)["AppliedToDocuments"].value },
                });

                paymentCreated = rs.Put("Payment", paymentAsString, null);
                if (paymentCreated != null)
                {

                    JObject payment = JObject.Parse(paymentCreated);

                    //Display the summary of the created payment
                    Console.WriteLine("Payment type: " + payment.GetValue("Type").Value<string>("value"));
                    Console.WriteLine("Payment number: " + payment.GetValue("ReferenceNbr").Value<string>("value"));
                    Console.WriteLine("Payment status: " + payment.GetValue("Status").Value<string>("value"));
                    Console.WriteLine("Payment amount: " + payment.GetValue("PaymentAmount").Value<string>("value"));
                    Console.WriteLine();
                    Console.WriteLine("Press any key to continue");
                    Console.ReadLine();
                }
            }        
        }

        //Capturing a payment
        public static void CapturePayment(RestService rs)
        {
            //Payment data
            string paymentType = "Payment";
            string paymentNbr = "000001"; //A payment that has On Hold status
            string parameters = "$expand=CreditCardProcessingInfo";

            //Convert the payment data to JSON format
            string paymentAsString = JsonConvert.SerializeObject(new
            {
                Type = new { value = paymentType },
                ReferenceNbr = new { value = paymentNbr },
                Hold = new { value = false }
            });

            //Clear the Hold checkbox
            string paymentToBeCaptured = rs.Put("Payment", paymentAsString, null);
            if (paymentToBeCaptured != null)
            {

                paymentAsString = JsonConvert.SerializeObject(new
                {
                    entity = JObject.Parse(paymentToBeCaptured)
                });

                //Capture the payment
                string paymentCaptured = rs.Post("Payment", "CaptureCreditCardPayment", paymentAsString);

                //Get the captured payment
                paymentCaptured = rs.GetByKeys("Payment", paymentType + '/' + paymentNbr, parameters);
                if (paymentCaptured != null)
                {
                    JObject payment = JObject.Parse(paymentCaptured);

                    //Display info on the captured payment
                    Console.WriteLine("Payment type: " + payment.GetValue("Type").Value<string>("value"));
                    Console.WriteLine("Payment number: " + payment.GetValue("ReferenceNbr").Value<string>("value"));
                    Console.WriteLine("Payment status: " + payment.GetValue("Status").Value<string>("value"));
                    Console.WriteLine("Transaction status: " + JsonConvert.DeserializeObject<dynamic>(paymentCaptured)["CreditCardProcessingInfo"][0]["TranStatus"].value);
                    Console.WriteLine("Transaction status: " + JsonConvert.DeserializeObject<dynamic>(paymentCaptured)["CreditCardProcessingInfo"][0]["TranAmount"].value);
                    Console.WriteLine();
                    Console.WriteLine("Press any key to continue");
                    Console.ReadLine();
                }
            }        
        }

        //Releasing a payment
        public static void ReleasePayment(RestService rs)
        {
            //Payment data
            string paymentType = "Payment";
            string paymentNbr = "000001"; //A payment that has the Balanced status

            //Convert the payment data to JSON
            string paymentAsString = JsonConvert.SerializeObject(new
            {
                entity = new
                {
                    Type = new { value = paymentType },
                    ReferenceNbr = new { value = paymentNbr }
                }
            });

            //Release the payment
            string paymentReleased = rs.Post("Payment", "ReleasePayment", paymentAsString);

            //Retrieve the payment
            paymentReleased = rs.GetByKeys("Payment", paymentType + '/' + paymentNbr, null);

            if (paymentReleased != null)
            {
                JObject payment = JObject.Parse(paymentReleased);

                //Display summary of the released payment
                Console.WriteLine("Payment type: " + payment.GetValue("Type").Value<string>("value"));
                Console.WriteLine("Payment number: " + payment.GetValue("ReferenceNbr").Value<string>("value"));
                Console.WriteLine("Payment status: " + payment.GetValue("Status").Value<string>("value"));
                Console.WriteLine();
                Console.WriteLine("Press any key to continue");
                Console.ReadLine();
            }        
        }

        //Retrieving the list of payments of a customer
        public static void ExportPayments(RestService rs)
        {
            //Input data
            string customerID = "C000000003";
            string docType = "Payment";

            //Filtering parameters
            string searchParameters = "$filter=Type eq '" + docType + "'&$filter=CustomerID eq '" + customerID + "'";

            //Export payments
            string paymentsExported = rs.Get("Payment", searchParameters);

            if (paymentsExported != null)
            {
                List<JObject> payments = JsonConvert.DeserializeObject<List<JObject>>(paymentsExported);

                //Save results to a CSV file
                using (StreamWriter file = new StreamWriter(
                string.Format(@"REST_Payments_Customer_{0}.csv", customerID)))
                {
                    //Add headers to the file
                    file.WriteLine("Type;ReferenceNbr;ApplicationDate;Status;");
                    //Write the values for each payment
                    foreach (JObject payment in payments)
                    {
                        file.WriteLine(string.Format("{0};{1};{2};{3};",
                        payment.GetValue("Type").Value<string>("value"),
                        payment.GetValue("ReferenceNbr").Value<string>("value"),
                        payment.GetValue("ApplicationDate").Value<string>("value"),
                        payment.GetValue("Status").Value<string>("value")));
                    }

                    file.Flush();
                }
            }
        }
    }
}
