﻿
using System.IO;
using System.Collections.Generic;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using MyStoreRestIntegration.Helpers;
using System;

namespace MyStoreRestIntegration.Integration
{
    public class SalesOrderProcessor
    {
        //Creating a sales order
        public static void CreateSO(RestService rs)
        {
            //Sales order data
            string orderType = "SO";
            string customerID = "C000000003";
            string orderDescription = "Test Sales Order";
            string customerOrder = "SO180-009-28"; //Any unique identifier
            string firstItemInventoryID = "AALEGO500";
            decimal firstItemQuantity = 10;
            string firstItemUOM = "PIECE";
            string secondItemInventoryID = "CONGRILL";
            decimal secondItemQuantity = 1;
            string secondItemUOM = "PIECE";

            //Convert the sales order data to JSON format
            string entityAsString = JsonConvert.SerializeObject(new
            {
                OrderType = new { value = orderType },
                CustomerID = new { value = customerID },
                Description = new { value = orderDescription },
                CustomerOrder = new { value = customerOrder },
                Details = new[]
                        {
                            new {
                            InventoryID = new {value = firstItemInventoryID },
                            Quantity = new {value = firstItemQuantity},
                            UOM = new {value = firstItemUOM}},

                            new {
                            InventoryID = new { value = secondItemInventoryID },
                            Quantity = new { value = secondItemQuantity },
                            UOM = new { value = secondItemUOM } }
                        }
            });

            //Create a sales order with the specified values
            string salesOrder = rs.Put("SalesOrder", entityAsString, null);

            if (salesOrder != null)
            {
                JObject salesOrderCreated = JObject.Parse(salesOrder);

                //Display the summary of the created sales order
                Console.WriteLine("Order type: " + salesOrderCreated.GetValue("OrderType").Value<string>("value"));
                Console.WriteLine("Order number: " + salesOrderCreated.GetValue("OrderNbr").Value<string>("value"));
                Console.WriteLine("Ordered quantity: " + salesOrderCreated.GetValue("OrderedQty").Value<string>("value"));
                Console.WriteLine("Order total: " + salesOrderCreated.GetValue("OrderTotal").Value<string>("value"));
                Console.WriteLine();
                Console.WriteLine("Press any key to continue");
                Console.ReadLine();
            }        
        }

        //Updating detail lines of a sales order
        public static void UpdateSO(RestService rs)
        {
            //Sales order data
            string orderType = "SO";
            string customerOrder = "SO248-563-06"; //A unique value that
            //identifies a sales order
            string firstItemInventoryID = "CONTABLE1";
            string firstItemWarehouse = "MAIN";
            string secondItemInventoryID = "AALEGO500";
            string secondItemWarehouse = "MAIN";
            decimal secondItemQuantity = 4;

            //Filtering parameters and the entity to be expanded
            string searchParameters = "$filter=CustomerOrder eq '" + customerOrder + "'&$filter=OrderType eq " + orderType + "&$expand=Details";

            //Find the sales order number to be updated
            string salesOrderToBeUpdated = rs.Get("SalesOrder", searchParameters);

            if (salesOrderToBeUpdated != null)
            {
                string salesOrderNbr = JsonConvert.DeserializeObject<dynamic>(salesOrderToBeUpdated)[0]["OrderNbr"].value;
                JObject jSalesOrder = JsonConvert.DeserializeObject<List<dynamic>>(salesOrderToBeUpdated)[0];
                JArray orderLines = jSalesOrder.Value<JArray>("Details");

                string detailLineId = null;

                //Specify the line to be deleted and mark it for deletion
                foreach (JObject orderLine in orderLines)
                {
                    string inventoryId = orderLine.GetValue("InventoryID").Value<string>("value");
                    string warehouseId = orderLine.GetValue("WarehouseID").Value<string>("value");
                    if (inventoryId == firstItemInventoryID && warehouseId == firstItemWarehouse)
                    {
                        detailLineId = orderLine.GetValue("id").ToString();
                        break;
                    }
                }

                string entityAsString = JsonConvert.SerializeObject(new
                {
                    OrderType = new { value = orderType },
                    OrderNbr = new { value = salesOrderNbr },
                    Details = new[]
                           {
                            new {
                            id = detailLineId,
                            delete =  true }
                        }
                });

                string salesOrderLineToBeDeleted = rs.Put("SalesOrder", entityAsString, null);

                //Specify the line to be updated and update quantity in it
                foreach (JObject orderLine in orderLines)
                {
                    string inventoryId = orderLine.GetValue("InventoryID").Value<string>("value");
                    string warehouseId = orderLine.GetValue("WarehouseID").Value<string>("value");
                    if (inventoryId == secondItemInventoryID && warehouseId == secondItemWarehouse)
                    {
                        detailLineId = orderLine.GetValue("id").ToString();
                        break;
                    }
                }

                entityAsString = JsonConvert.SerializeObject(new
                {
                    OrderType = new { value = orderType },
                    OrderNbr = new { value = salesOrderNbr },
                    Hold = new { value = false },
                    Details = new[]
                            {
                            new {
                            id = detailLineId,
                            Quantity =  new { value = secondItemQuantity} }
                        }
                });

                string salesOrderLineUpdated = rs.Put("SalesOrder", entityAsString, null);


                if (salesOrderLineUpdated != null)
                {
                    JObject salesOrderUpdated = JObject.Parse(salesOrderLineUpdated);

                    //Display the summary of the updated record
                    Console.WriteLine("Order type: " + salesOrderUpdated.GetValue("OrderType").Value<string>("value"));
                    Console.WriteLine("Order number: " + salesOrderUpdated.GetValue("OrderNbr").Value<string>("value"));
                    Console.WriteLine("Ordered quantity: " + salesOrderUpdated.GetValue("OrderedQty").Value<string>("value"));
                    Console.WriteLine("Order total: " + salesOrderUpdated.GetValue("OrderTotal").Value<string>("value"));
                    Console.WriteLine();
                    Console.WriteLine("Press any key to continue");
                    Console.ReadLine();
                }
            }       
        }

        //Retrieving details of a sales order by the order number
        public static void ExportSODetails(RestService rs)
        {
            //Sales order data
            string orderType = "SO";
            string orderNbr = "000001";
            string parameters = "$expand=Details";

            //Get the sales order
            string salesOrder = rs.GetByKeys("SalesOrder", orderType + '/' + orderNbr, parameters);

            if (salesOrder != null)
            {
                JObject jSalesOrder = JObject.Parse(salesOrder);
                JArray orderLines = jSalesOrder.Value<JArray>("Details");

                //Save results to a CSV file
                using (StreamWriter file = new StreamWriter(
                string.Format(@"REST_SalesOrder_{0}.csv", orderNbr)))
                {
                    //Add headers to the file
                    file.WriteLine("OrderType;OrderNbr;CustomerID;CustomerOrder;" +
                    "Date;OrderedQty;OrderTotal;TaxTotal;InventoryID;UOM;" +
                    "Quantity;UnitPrice");
                    //Write the values for each detail line
                    foreach (JObject orderLine in orderLines)
                    {
                        file.WriteLine(string.Format(
                        "{0};{1};{2};{3};{4};{5};{6};{7};{8};{9};{10};{11}",
                            //Document summary
                        jSalesOrder.GetValue("OrderType").Value<string>("value"),
                        jSalesOrder.GetValue("OrderNbr").Value<string>("value"),
                        jSalesOrder.GetValue("CustomerID").Value<string>("value"),
                        jSalesOrder.GetValue("CustomerOrder").Value<string>("value"),
                        jSalesOrder.GetValue("Date").Value<string>("value"),
                        jSalesOrder.GetValue("OrderedQty").Value<string>("value"),
                        jSalesOrder.GetValue("OrderTotal").Value<string>("value"),
                            //Totals
                        jSalesOrder.GetValue("TaxTotal").Value<string>("value"),
                            //Document details
                        orderLine.GetValue("InventoryID").Value<string>("value"),
                        orderLine.GetValue("UOM").Value<string>("value"),
                        orderLine.GetValue("Quantity").Value<string>("value"),
                        orderLine.GetValue("UnitPrice").Value<string>("value")));
                    }
                    file.Flush();
                }
            }        
        }


        //Retrieving the list of sales order of a customer
        public static void ExportSalesOrders(RestService rs)
        {
            //Customer data
            string customerID = "C000000003";

            //Filtering parameters
            string searchparameter = "$filter=CustomerID eq '" + customerID + "'";

            //Retrieve the list of sales orders of the customer
            string salesOrders = rs.Get("SalesOrder", searchparameter);
            if (salesOrders != null)
            {
                var salesOrderList = JsonConvert.DeserializeObject<List<dynamic>>(salesOrders);

                //Save results to a CSV file
                using (StreamWriter file = new StreamWriter(
                string.Format(@"REST_SalesOrders_Customer_{0}.csv", customerID)))
                {
                    //Add headers to the file
                    file.WriteLine("OrderType;OrderNbr;CustomerID;CustomerOrder;" +
                    "Date;OrderedQty;OrderTotal;");
                    //Write the values for each sales order
                    foreach (JObject salesOrder in salesOrderList)
                    {
                        file.WriteLine(string.Format("{0};{1};{2};{3};{4};{5};{6}",
                        salesOrder.GetValue("OrderType").Value<string>("value"),
                        salesOrder.GetValue("OrderNbr").Value<string>("value"),
                        salesOrder.GetValue("CustomerID").Value<string>("value"),
                        salesOrder.GetValue("CustomerOrder").Value<string>("value"),
                        salesOrder.GetValue("Date").Value<string>("value"),
                        salesOrder.GetValue("OrderedQty").Value<string>("value"),
                        salesOrder.GetValue("OrderTotal").Value<string>("value")));
                    }

                    file.Flush();
                }
            }        
        }

    }
}
